<?php
// /home/ubuntu/vtrius_web_system/src/Lib/Database.php

namespace Vtrius\Lib;

use PDO;
use PDOException;

class Database {
    private static ?PDO $pdo = null;
    private static string $dbPath = '';

    // Private constructor to prevent direct instantiation
    private function __construct() {}

    // Prevent cloning
    private function __clone() {}

    // Prevent unserialization
    public function __wakeup() {}

    /**
     * Initializes the database path and ensures the directory is writable.
     * Should be called once at the beginning, e.g., from config or index.php.
     *
     * @param string $path The absolute path to the SQLite database file.
     * @throws \Exception If the database directory is not writable.
     */
    public static function init(string $path): void {
        self::$dbPath = $path;
        $dir = dirname(self::$dbPath);

        if (!is_dir($dir)) {
            if (!mkdir($dir, 0755, true)) {
                 throw new \Exception("Failed to create database directory: {$dir}");
            }
        }

        if (!is_writable($dir)) {
            // Attempt to make it writable (might fail depending on server setup)
            @chmod($dir, 0755);
            if (!is_writable($dir)) {
                throw new \Exception("Database directory '{$dir}' is not writable. Please check permissions.");
            }
        }
    }

    /**
     * Gets the PDO database connection instance (Singleton pattern).
     *
     * @return PDO The PDO connection instance.
     * @throws \Exception If Database::init() was not called or connection fails.
     */
    public static function getConnection(): PDO {
        if (empty(self::$dbPath)) {
            throw new \Exception("Database path not initialized. Call Database::init() first.");
        }

        if (self::$pdo === null) {
            $dsn = "sqlite:" . self::$dbPath;
            try {
                self::$pdo = new PDO($dsn);
                self::$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                self::$pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
                // Enable foreign key constraints for SQLite
                self::$pdo->exec('PRAGMA foreign_keys = ON;');
                self::initializeDatabase(self::$pdo);
            } catch (PDOException $e) {
                // Log the error instead of dying
                error_log("Database Connection/Initialization Error: " . $e->getMessage());
                throw new \Exception("Database Connection Error. Check logs for details.");
            }
        }
        return self::$pdo;
    }

    /**
     * Initializes the database schema if tables don't exist.
     * Reads the schema from db/schema.sql.
     *
     * @param PDO $pdo The PDO connection instance.
     * @throws \Exception If schema file is not found or readable.
     */
    private static function initializeDatabase(PDO $pdo): void {
        $schemaPath = dirname(__DIR__, 2) . '/db/schema.sql'; // Assumes db/ is sibling to src/
        if (!file_exists($schemaPath) || !is_readable($schemaPath)) {
            throw new \Exception("Database schema file not found or not readable: {$schemaPath}");
        }
        try {
            $schemaSql = file_get_contents($schemaPath);
            // Execute the schema SQL (might contain multiple statements)
            $pdo->exec($schemaSql);
        } catch (PDOException $e) {
            error_log("Error executing database schema: " . $e->getMessage());
            throw new \Exception("Error initializing database schema. Check logs.");
        }
    }
}

