<?php
// /home/ubuntu/vtrius_web_system/src/Lib/I18n.php

namespace Vtrius\Lib;

class I18n {
    private static string $defaultLang = "pt_BR";
    private static array $supportedLangs = ["pt_BR", "en_US", "es_ES"];
    private static string $currentLang = "";
    private static array $translations = [];

    /**
     * Initialize the I18n system.
     * Determines the current language based on session, query param, or default.
     */
    public static function init(): void {
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }

        // 1. Check for language change via query parameter (e.g., ?lang=en_US)
        if (isset($_GET["lang"]) && in_array($_GET["lang"], self::$supportedLangs)) {
            self::$currentLang = $_GET["lang"];
            $_SESSION["vtrius_lang"] = self::$currentLang;
            // Redirect to remove the lang parameter from URL after setting session
            $uriWithoutQuery = strtok($_SERVER["REQUEST_URI"], 
            "?
            ");
            header(
            "Location: 
            " . $uriWithoutQuery);
            exit;
        } 
        // 2. Check if language is already set in session
        elseif (isset($_SESSION["vtrius_lang"]) && in_array($_SESSION["vtrius_lang"], self::$supportedLangs)) {
            self::$currentLang = $_SESSION["vtrius_lang"];
        } 
        // 3. TODO: Optionally detect browser language (Accept-Language header)
        // else if (isset($_SERVER["HTTP_ACCEPT_LANGUAGE"])) { ... }
        
        // 4. Fallback to default language
        else {
            self::$currentLang = self::$defaultLang;
        }

        self::loadTranslations();
    }

    /**
     * Load translation strings for the current language.
     */
    private static function loadTranslations(): void {
        $langFile = APP_ROOT . 
        "/Locale/
        " . self::$currentLang . 
        "/messages.php
        ";

        if (file_exists($langFile)) {
            self::$translations = require $langFile;
        } else {
            // Attempt to load default language if current language file is missing
            $defaultLangFile = APP_ROOT . 
            "/Locale/
            " . self::$defaultLang . 
            "/messages.php
            ";
            if (self::$currentLang !== self::$defaultLang && file_exists($defaultLangFile)) {
                self::$translations = require $defaultLangFile;
                error_log(
                "Warning: Language file for 
                " . self::$currentLang . 
                " not found. Falling back to 
                " . self::$defaultLang . 
                ".
                ");
            } else {
                self::$translations = []; // No translations available
                error_log(
                "Error: Could not load language file for 
                " . self::$currentLang . 
                " or default language 
                " . self::$defaultLang . 
                ".
                ");
            }
        }
    }

    /**
     * Get the translation for a given key.
     *
     * @param string $key The translation key.
     * @param array $params Optional parameters for placeholder replacement (e.g., ["name" => "John"]).
     * @return string The translated string or the key itself if not found.
     */
    public static function translate(string $key, array $params = []): string {
        $text = self::$translations[$key] ?? $key; // Return key if translation not found

        // Replace placeholders like :name
        if (!empty($params)) {
            foreach ($params as $placeholder => $value) {
                $text = str_replace(
                ":
                " . $placeholder, $value, $text);
            }
        }
        return $text;
    }

    /**
     * Get the currently active language code.
     *
     * @return string
     */
    public static function getCurrentLang(): string {
        return self::$currentLang;
    }

    /**
     * Get the list of supported language codes.
     *
     * @return array
     */
    public static function getSupportedLangs(): array {
        return self::$supportedLangs;
    }
}

// Initialize I18n (e.g., call this early in your config.php or index.php)
// I18n::init();

// Global helper function for easier access in views
function __(string $key, array $params = []): string {
    return I18n::translate($key, $params);
}


