<?php
// /home/ubuntu/vtrius_web_system/src/Models/AbTestVariation.php

namespace Vtrius\Models;

use Vtrius\Lib\Database;
use PDO;

class AbTestVariation {
    private PDO $db;

    public function __construct() {
        $this->db = Database::getConnection();
    }

    /**
     * Add a video variation to an A/B test.
     *
     * @param int $abTestId
     * @param int $videoId
     * @param bool $isControl Whether this variation is the control.
     * @return int|false The ID of the newly created variation or false on failure.
     */
    public function addVariation(int $abTestId, int $videoId, bool $isControl = false): int|false {
        // Check if this video is already part of the test
        $stmtCheck = $this->db->prepare("SELECT id FROM ab_variations WHERE ab_test_id = :ab_test_id AND video_id = :video_id");
        $stmtCheck->bindParam(":ab_test_id", $abTestId, PDO::PARAM_INT);
        $stmtCheck->bindParam(":video_id", $videoId, PDO::PARAM_INT);
        $stmtCheck->execute();
        if ($stmtCheck->fetch()) {
            // Variation already exists for this test
            return false; 
        }

        // Check variation limit (e.g., max 4 per test including control)
        $stmtCount = $this->db->prepare("SELECT COUNT(*) FROM ab_variations WHERE ab_test_id = :ab_test_id");
        $stmtCount->bindParam(":ab_test_id", $abTestId, PDO::PARAM_INT);
        $stmtCount->execute();
        $count = $stmtCount->fetchColumn();
        if ($count >= 4) {
            // Limit reached
            error_log("A/B Test Variation Limit Reached for test ID: " . $abTestId);
            return false;
        }

        $sql = "INSERT INTO ab_variations (ab_test_id, video_id, is_control) VALUES (:ab_test_id, :video_id, :is_control)";
        $stmt = $this->db->prepare($sql);
        $isControlInt = $isControl ? 1 : 0;
        $stmt->bindParam(":ab_test_id", $abTestId, PDO::PARAM_INT);
        $stmt->bindParam(":video_id", $videoId, PDO::PARAM_INT);
        $stmt->bindParam(":is_control", $isControlInt, PDO::PARAM_INT);

        if ($stmt->execute()) {
            return (int)$this->db->lastInsertId();
        } else {
            return false;
        }
    }

    /**
     * Find all variations for a specific A/B test.
     *
     * @param int $abTestId
     * @return array List of variations with video details.
     */
    public function findByTestId(int $abTestId): array {
        // Join with videos table to get video titles
        $sql = "SELECT v.*, av.id as variation_id, av.is_control 
                FROM ab_variations av
                JOIN videos v ON av.video_id = v.id
                WHERE av.ab_test_id = :ab_test_id
                ORDER BY av.is_control DESC, v.created_at ASC"; // Show control first
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":ab_test_id", $abTestId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * Remove a variation from an A/B test.
     * Cannot remove the control variation directly (should delete the test or change control).
     *
     * @param int $variationId
     * @param int $abTestId To ensure it belongs to the correct test (optional check).
     * @return bool True on success, false on failure.
     */
    public function removeVariation(int $variationId, int $abTestId): bool {
        // Prevent deleting the control variation
        $stmtCheck = $this->db->prepare("SELECT is_control FROM ab_variations WHERE id = :id AND ab_test_id = :ab_test_id");
        $stmtCheck->bindParam(":id", $variationId, PDO::PARAM_INT);
        $stmtCheck->bindParam(":ab_test_id", $abTestId, PDO::PARAM_INT);
        $stmtCheck->execute();
        $variation = $stmtCheck->fetch();

        if (!$variation || $variation["is_control"]) {
            return false; // Not found or is control
        }

        // Proceed with deletion (CASCADE should handle stats)
        $sql = "DELETE FROM ab_variations WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":id", $variationId, PDO::PARAM_INT);
        return $stmt->execute();
    }

    /**
     * Get a random variation ID for a given test ID.
     *
     * @param int $abTestId
     * @return int|false The ID of a randomly selected variation or false if none found.
     */
    public function getRandomVariationId(int $abTestId): int|false {
        $stmt = $this->db->prepare("SELECT id FROM ab_variations WHERE ab_test_id = :ab_test_id ORDER BY RANDOM() LIMIT 1");
        $stmt->bindParam(":ab_test_id", $abTestId, PDO::PARAM_INT);
        $stmt->execute();
        $result = $stmt->fetchColumn();
        return $result ? (int)$result : false;
    }
    
    /**
     * Find variation details by its ID.
     *
     * @param int $variationId
     * @return array|false Variation data including video_id and ab_test_id.
     */
    public function findById(int $variationId): array|false {
        $stmt = $this->db->prepare("SELECT * FROM ab_variations WHERE id = :id");
        $stmt->bindParam(":id", $variationId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    }

    // TODO: Add methods for recording stats (views, completions)
    // public function recordView(int $variationId) { ... }
    // public function recordCompletion(int $variationId) { ... }
    // public function getStats(int $abTestId) { ... }
}

