<?php
// /home/ubuntu/vtrius_web_system/src/Models/Video.php

namespace Vtrius\Models;

use Vtrius\Lib\Database;
use PDO;

class Video {
    private PDO $db;

    public function __construct() {
        $this->db = Database::getConnection();
    }

    /**
     * Create a new video record.
     *
     * @param int $userId
     * @param string $title
     * @param string $originalFilename
     * @param string $storagePath Relative path where the video is stored.
     * @param string $mimeType
     * @param int $filesize
     * @return int|false The ID of the newly created video or false on failure.
     */
    public function create(int $userId, string $title, string $originalFilename, string $storagePath, string $mimeType, int $filesize): int|false {
        $sql = "INSERT INTO videos (user_id, title, original_filename, storage_path, mime_type, filesize) 
                VALUES (:user_id, :title, :original_filename, :storage_path, :mime_type, :filesize)";
        $stmt = $this->db->prepare($sql);

        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        $stmt->bindParam(":title", $title);
        $stmt->bindParam(":original_filename", $originalFilename);
        $stmt->bindParam(":storage_path", $storagePath);
        $stmt->bindParam(":mime_type", $mimeType);
        $stmt->bindParam(":filesize", $filesize, PDO::PARAM_INT);

        if ($stmt->execute()) {
            $videoId = (int)$this->db->lastInsertId();
            // Automatically create default settings for the new video
            $videoSettingModel = new VideoSetting();
            $videoSettingModel->createDefaults($videoId);
            return $videoId;
        }
        return false;
    }

    /**
     * Find a video by its ID.
     *
     * @param int $id
     * @return array|false Video data or false if not found.
     */
    public function findById(int $id): array|false {
        $stmt = $this->db->prepare("SELECT * FROM videos WHERE id = :id");
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    }

    /**
     * Find a video by its ID and user ID (for ownership verification).
     *
     * @param int $id
     * @param int $userId
     * @return array|false Video data or false if not found/owned.
     */
    public function findByIdAndUser(int $id, int $userId): array|false {
        $stmt = $this->db->prepare("SELECT * FROM videos WHERE id = :id AND user_id = :user_id");
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    }

    /**
     * Get all videos belonging to a specific user.
     *
     * @param int $userId
     * @return array List of videos.
     */
    public function findByUser(int $userId): array {
        // Join with settings to potentially display more info in the list later
        $sql = "SELECT v.*, vs.hide_controls, vs.overlay_image_path, vs.pause_cover_image_path, vs.pitch_time_seconds 
                FROM videos v
                LEFT JOIN video_settings vs ON v.id = vs.video_id
                WHERE v.user_id = :user_id 
                ORDER BY v.created_at DESC";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * Update video details (e.g., title).
     *
     * @param int $id
     * @param string $title
     * @return bool True on success, false on failure.
     */
    public function updateTitle(int $id, string $title): bool {
        $sql = "UPDATE videos SET title = :title WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->bindParam(":title", $title);
        return $stmt->execute();
    }

    /**
     * Delete a video record (and associated settings via CASCADE).
     *
     * @param int $id
     * @param int $userId For ownership verification.
     * @return bool True on success, false on failure or if not found/owned.
     */
    public function delete(int $id, int $userId): bool {
        // First, verify ownership
        $video = $this->findByIdAndUser($id, $userId);
        if (!$video) {
            return false; // Not found or not owned by the user
        }

        // Proceed with deletion (CASCADE should handle settings, AB variations, etc.)
        $sql = "DELETE FROM videos WHERE id = :id AND user_id = :user_id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        
        if ($stmt->execute()) {
            // Also delete the physical file
            $filePath = PROJECT_ROOT . 
            '/public/uploads/videos/
            ' . $userId . 
            '/
            ' . basename($video[
            'storage_path
            ']);
            if (file_exists($filePath)) {
                @unlink($filePath);
            }
            // Optionally delete related overlay/pause cover images here too
            // ... (code to find and delete associated images)
            return true;
        }
        return false;
    }
    
    /**
     * Count videos for a specific user.
     *
     * @param int $userId
     * @return int
     */
    public function countByUser(int $userId): int {
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM videos WHERE user_id = :user_id");
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        $stmt->execute();
        return (int)$stmt->fetchColumn();
    }

}

