<?php
// /home/ubuntu/vtrius_web_system/public/api/abtest_data.php

// This endpoint provides the data for a randomly selected variation of an A/B test.
// It's called by the ab_player.js embed script.

require_once dirname(__DIR__) . 
"/../src/config.php
"; // Adjust path as needed

use Vtrius\Models\AbTest;
use Vtrius\Models\AbTestVariation;
use Vtrius\Models\Video;
use Vtrius\Models\VideoSetting;

header(
"Content-Type: application/json
");
header(
"Access-Control-Allow-Origin: *
"); // Allow requests from any origin (adjust for production)

$testId = filter_input(INPUT_GET, 
"test
", FILTER_VALIDATE_INT);

if (!$testId) {
    http_response_code(400);
    echo json_encode([
    "error
    " => 
    "Invalid or missing test ID.
    "]);
    exit;
}

try {
    $abTestModel = new AbTest();
    $abVariationModel = new AbTestVariation();
    $videoModel = new Video();
    $videoSettingModel = new VideoSetting();

    // 1. Get a random variation ID for the test
    $variationId = $abVariationModel->getRandomVariationId($testId);

    if (!$variationId) {
        http_response_code(404);
        echo json_encode([
        "error
        " => 
        "Test not found or has no variations.
        "]);
        exit;
    }

    // 2. Get the variation details (including video_id)
    $variation = $abVariationModel->findById($variationId);
    if (!$variation) {
        // Should not happen if getRandomVariationId worked, but check anyway
        http_response_code(404);
        echo json_encode([
        "error
        " => 
        "Variation details not found.
        "]);
        exit;
    }
    $videoId = $variation[
    "video_id
    "];

    // 3. Get the video details
    // Note: We don't check ownership here as the test ID implies access
    $video = $videoModel->findById($videoId); 
    if (!$video) {
        http_response_code(404);
        echo json_encode([
        "error
        " => 
        "Video associated with variation not found.
        "]);
        exit;
    }

    // 4. Get the video settings
    $settings = $videoSettingModel->findByVideoId($videoId);
    if (!$settings) {
        // Create defaults if missing (should ideally be created on video upload/setting view)
        $videoSettingModel->createDefaults($videoId);
        $settings = $videoSettingModel->findByVideoId($videoId);
    }

    // 5. Prepare the data for the player
    $playerData = [
        
        "variationId
        " => $variationId, // Include variation ID for tracking
        
        "videoId
        " => $videoId,
        
        "videoUrl
        " => BASE_URL . 
        "/uploads/videos/
        " . htmlspecialchars($video[
        "storage_path
        "]), 
        
        "hideControls
        " => (bool)$settings[
        "hide_controls
        "],
        
        "autoplay
        " => (bool)$settings[
        "autoplay
        "],
        
        "overlayImageUrl
        " => $settings[
        "overlay_image_path
        "] ? BASE_URL . 
        "/uploads/overlays/
        " . htmlspecialchars($settings[
        "overlay_image_path
        "]) : null,
        
        "overlayOpacity
        " => (float)($settings[
        "overlay_opacity
        "] ?? 1.0),
        
        "pauseCoverImageUrl
        " => $settings[
        "pause_cover_image_path
        "] ? BASE_URL . 
        "/uploads/pause_covers/
        " . htmlspecialchars($settings[
        "pause_cover_image_path
        "]) : null,
        
        "pauseCoverText
        " => $settings[
        "pause_cover_text
        "] ?? 
        "Você já estava assistindo...
        ", // TODO: Use i18n based on user/site settings?
        
        "pitchTimeSeconds
        " => $settings[
        "pitch_time_seconds
        "] ? (int)$settings[
        "pitch_time_seconds
        "] : null,
        
        "pitchTargetSelector
        " => $settings[
        "pitch_target_selector
        "] ?? null,
        
        "pitchButtonText
        " => $settings[
        "pitch_button_text
        "] ?? null,
        
        "pitchButtonUrl
        " => $settings[
        "pitch_button_url
        "] ?? null,
        // Add translated button texts if needed by JS player
        // "textContinue" => __("pause_cover_continue_button"),
        // "textRestart" => __("pause_cover_restart_button"),
    ];

    // 6. Return JSON response
    echo json_encode($playerData);

} catch (\Exception $e) {
    error_log(
    "Error in abtest_data.php: 
    " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
    "error
    " => 
    "An internal server error occurred.
    "]);
}

