<?php
// /home/ubuntu/vtrius_web_system/src/Controllers/VideoSettingsController.php

namespace Vtrius\Controllers;

use Vtrius\Lib\Auth;
use Vtrius\Models\Video;
use Vtrius\Models\VideoSetting;
use Vtrius\Lib\Utils; // For file uploads

class VideoSettingsController {
    private Auth $auth;
    private Video $videoModel;
    private VideoSetting $videoSettingModel;
    private ?int $userId;

    public function __construct() {
        // Ensure config is loaded
        $this->auth = new Auth();
        $this->auth->requireLogin(BASE_URL . 
        "/login.php
        ");
        $this->userId = $this->auth->getUserId();
        $this->videoModel = new Video();
        $this->videoSettingModel = new VideoSetting();
    }

    /**
     * Display the settings form for a specific video.
     */
    public function showSettingsForm(): void {
        $videoId = filter_input(INPUT_GET, 
        "id
        ", FILTER_VALIDATE_INT);

        if (!$videoId) {
            $this->flashMessage(__("invalid_request"), 
            "error
            ");
            $this->redirect(BASE_URL . 
            "/user/videos.php
            ");
        }

        // Verify video ownership
        $video = $this->videoModel->findByIdAndUser($videoId, $this->userId);
        if (!$video) {
            $this->flashMessage(__("video_delete_error"), 
            "error
            "); // Reusing delete error message for permission denied
            $this->redirect(BASE_URL . 
            "/user/videos.php
            ");
        }

        // Get current settings, create defaults if they don't exist
        $settings = $this->videoSettingModel->findByVideoId($videoId);
        if (!$settings) {
            $this->videoSettingModel->createDefaults($videoId);
            $settings = $this->videoSettingModel->findByVideoId($videoId); // Fetch again
        }

        $pageTitleKey = 
        "video_settings_title
        "; // Needs translation
        $this->loadView(
        "user/videos/settings
        ", [
            
            "pageTitleKey
            " => $pageTitleKey,
            
            "video
            " => $video,
            
            "settings
            " => $settings
        ]);
    }

    /**
     * Handle the submission of the video settings form.
     */
    public function handleUpdateSettings(): void {
        if ($_SERVER[
        "REQUEST_METHOD
        "] !== 
        "POST
        ") {
            http_response_code(405);
            echo __("method_not_allowed");
            exit;
        }

        $videoId = filter_input(INPUT_POST, 
        "video_id
        ", FILTER_VALIDATE_INT);

        if (!$videoId) {
            $this->flashMessage(__("invalid_request"), 
            "error
            ");
            $this->redirect(BASE_URL . 
            "/user/videos.php
            ");
        }

        // Verify video ownership
        $video = $this->videoModel->findByIdAndUser($videoId, $this->userId);
        if (!$video) {
            $this->flashMessage(__("video_delete_error"), 
            "error
            ");
            $this->redirect(BASE_URL . 
            "/user/videos.php
            ");
        }

        // --- Process Form Data ---
        $updateData = [];
        $updateData[
        "hide_controls
        "] = isset($_POST[
        "hide_controls
        "]) ? 1 : 0;
        $updateData[
        "autoplay
        "] = isset($_POST[
        "autoplay
        "]) ? 1 : 0; // Add to DB schema if not present
        $updateData[
        "overlay_opacity
        "] = filter_input(INPUT_POST, 
        "overlay_opacity
        ", FILTER_VALIDATE_FLOAT, [
        "options
        " => [
        "min_range
        " => 0, 
        "max_range
        " => 1, 
        "default
        " => 1.0]]);
        $updateData[
        "pause_cover_text
        "] = trim($_POST[
        "pause_cover_text
        "] ?? 
        "" );
        $updateData[
        "pitch_time_seconds
        "] = filter_input(INPUT_POST, 
        "pitch_time_seconds
        ", FILTER_VALIDATE_INT);
        if ($updateData[
        "pitch_time_seconds
        "] === false || $updateData[
        "pitch_time_seconds
        "] < 0) {
            $updateData[
            "pitch_time_seconds
            "] = null; // Store as NULL if invalid or empty
        }
        $updateData[
        "pitch_target_sele        $updateData["pitch_target_selector"] = trim($_POST["pitch_target_selector"] ?? "");
        $updateData["pitch_button_text"] = trim($_POST["pitch_button_text"] ?? "");
        $updateData["pitch_button_url"] = filter_input(INPUT_POST, "pitch_button_url", FILTER_VALIDATE_URL) ?: null; // Store as NULL if invalid or empty

        // --- Handle Image Uploads ---        $currentSettings = $this->videoSettingModel->findByVideoId($videoId);

        // Overlay Image
        $overlayImageResult = $this->handleImageUpload(
        "overlay_image
        ", OVERLAYS_DIR, $currentSettings[
        "overlay_image_path
        "] ?? null);
        if ($overlayImageResult[
        "status
        "] === 
        "success
        ") {
            $updateData[
            "overlay_image_path
            "] = $overlayImageResult[
            "path
            "];
        } elseif ($overlayImageResult[
        "status
        "] === 
        "error
        ") {
            $this->flashMessage($overlayImageResult[
            "message
            "], 
            "error
            ");
            $this->redirect(BASE_URL . 
            "/user/videos/settings.php?id=
            " . $videoId);
        }

        // Pause Cover Image
        $pauseCoverResult = $this->handleImageUpload(
        "pause_cover_image
        ", PAUSE_COVERS_DIR, $currentSettings[
        "pause_cover_image_path
        "] ?? null);
        if ($pauseCoverResult[
        "status
        "] === 
        "success
        ") {
            $updateData[
            "pause_cover_image_path
            "] = $pauseCoverResult[
            "path
            "];
        } elseif ($pauseCoverResult[
        "status
        "] === 
        "error
        ") {
            $this->flashMessage($pauseCoverResult[
            "message
            "], 
            "error
            ");
            $this->redirect(BASE_URL . 
            "/user/videos/settings.php?id=
            " . $videoId);
        }

        // --- Update Database ---
        if ($this->videoSettingModel->update($videoId, $updateData)) {
            $this->flashMessage(__("video_settings_update_success"), 
            "success
            "); // Needs translation
        } else {
            $this->flashMessage(__("video_settings_update_error"), 
            "error
            "); // Needs translation
        }

        $this->redirect(BASE_URL . 
        "/user/videos/settings.php?id=
        " . $videoId);
    }

    /**
     * Helper function to handle image uploads for settings.
     *
     * @param string $fileInputName The name of the file input field.
     * @param string $destinationDir The absolute path to the destination directory.
     * @param ?string $currentPath The current relative path stored in DB (for deletion).
     * @return array Status array ["status" => "success|error|no_upload", "path" => "relative/path/to/file.ext", "message" => "Error message"].
     */
    private function handleImageUpload(string $fileInputName, string $destinationDir, ?string $currentPath): array {
        $file = $_FILES[$fileInputName] ?? null;

        // Check if a file was uploaded and there are no errors
        if (!$file || $file[
        "error
        "] === UPLOAD_ERR_NO_FILE) {
            // Check if user wants to remove the existing image
            if (isset($_POST["remove_" . $fileInputName]) && $currentPath) {
                $fullPath = $destinationDir . 
                "/
                " . $currentPath;
                @unlink($fullPath);
                return [
                "status
                " => 
                "success
                ", 
                "path
                " => null, 
                "message
                " => 
                "Image removed.
                "]; // Return null path
            }
            return [
            "status
            " => 
            "no_upload
            ", 
            "path
            " => $currentPath, 
            "message
            " => 
            "No file uploaded.
            "]; // No change
        }

        if ($file[
        "error
        "] !== UPLOAD_ERR_OK) {
            return [
            "status
            " => 
            "error
            ", 
            "path
            " => $currentPath, 
            "message
            " => __("upload_error_generic", [
            "message
            " => $this->getUploadErrorMessage($file[
            "error
            "])])];
        }

        // Validate MIME type
        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($file[
        "tmp_name
        "]);
        if (!in_array($mimeType, ALLOWED_IMAGE_MIMES)) {
            return [
            "status
            " => 
            "error
            ", 
            "path
            " => $currentPath, 
            "message
            " => __("upload_error_image_type")]; // Needs translation
        }

        // Validate size (e.g., max 5MB for images)
        $maxImageSize = 5 * 1024 * 1024;
        if ($file[
        "size
        "] > $maxImageSize) {
             return [
             "status
             " => 
             "error
             ", 
             "path
             " => $currentPath, 
             "message
             " => __("upload_error_image_size", [
             "size
             " => 5])]; // Needs translation
        }

        // Create unique filename
        $extension = pathinfo($file[
        "name
        "], PATHINFO_EXTENSION);
        $uniqueFilename = uniqid(pathinfo($fileInputName, PATHINFO_FILENAME) . 
        "_
        ", true) . 
        ".
        " . strtolower($extension);
        $destinationPath = $destinationDir . 
        "/
        " . $uniqueFilename;

        // Ensure destination directory exists
        if (!is_dir($destinationDir)) {
            if (!mkdir($destinationDir, 0755, true)) {
                return [
                "status
                " => 
                "error
                ", 
                "path
                " => $currentPath, 
                "message
                " => __("upload_error_cannot_create_dir")];
            }
        }

        // Move the file
        if (move_uploaded_file($file[
        "tmp_name
        "], $destinationPath)) {
            // Delete old file if it exists and is different
            if ($currentPath && $currentPath !== $uniqueFilename) {
                $oldFullPath = $destinationDir . 
                "/
                " . $currentPath;
                @unlink($oldFullPath);
            }
            return [
            "status
            " => 
            "success
            ", 
            "path
            " => $uniqueFilename, 
            "message
            " => 
            "Upload successful.
            "];
        } else {
            return [
            "status
            " => 
            "error
            ", 
            "path
            " => $currentPath, 
            "message
            " => __("upload_error_cannot_move_file")];
        }
    }

    // --- Helper Methods (Copied from VideoController, consider a BaseController) ---

    private function loadView(string $viewName, array $data = []): void {
        extract($data);
        $viewPath = APP_ROOT . 
        "/Views/
        " . $viewName . 
        ".php
        ";
        if (file_exists($viewPath)) {
            $headerPath = APP_ROOT . 
            "/Views/layouts/user_header.php
            ";
            $footerPath = APP_ROOT . 
            "/Views/layouts/user_footer.php
            ";
            include $headerPath;
            include $viewPath;
            include $footerPath;
        } else {
            http_response_code(500);
            echo __("view_not_found", [
            "path
            " => $viewPath]);
        }
    }

    private function redirect(string $url): void {
        header(
        "Location: 
        " . $url);
        exit;
    }

    private function flashMessage(string $messageOrKey, string $type = 
    "success
    ", array $params = []): void {
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
        // Store key and params for translation in the view
        $_SESSION[
        "flash_message
        "] = [
        "key
        " => $messageOrKey, 
        "params
        " => $params, 
        "type
        " => $type];
    }

    private function getUploadErrorMessage(int $errorCode): string {
        // Use translation keys directly
        return match ($errorCode) {
            UPLOAD_ERR_INI_SIZE => __("upload_err_ini_size"),
            UPLOAD_ERR_FORM_SIZE => __("upload_err_form_size"),
            UPLOAD_ERR_PARTIAL => __("upload_err_partial"),
            UPLOAD_ERR_NO_FILE => __("upload_err_no_file"),
            UPLOAD_ERR_NO_TMP_DIR => __("upload_err_no_tmp_dir"),
            UPLOAD_ERR_CANT_WRITE => __("upload_err_cant_write"),
            UPLOAD_ERR_EXTENSION => __("upload_err_extension"),
            default => __("upload_err_unknown"),
        };
    }
}

