<?php
// /home/ubuntu/vtrius_web_system/src/Models/User.php

namespace Vtrius\Models;

use Vtrius\Lib\Database;
use PDO;
use DateTime;

class User {
    private PDO $db;

    public function __construct() {
        $this->db = Database::getConnection();
    }

    /**
     * Find a user by their username.
     *
     * @param string $username
     * @return array|false User data as an associative array or false if not found.
     */
    public function findByUsername(string $username) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE username = :username");
        $stmt->bindParam(":username", $username);
        $stmt->execute();
        return $stmt->fetch();
    }

    /**
     * Find a user by their email.
     *
     * @param string $email
     * @return array|false User data as an associative array or false if not found.
     */
    public function findByEmail(string $email) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE email = :email");
        $stmt->bindParam(":email", $email);
        $stmt->execute();
        return $stmt->fetch();
    }

    /**
     * Find a user by their ID.
     *
     * @param int $id
     * @return array|false User data as an associative array or false if not found.
     */
    public function findById(int $id) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE id = :id");
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    }

    /**
     * Create a new user.
     *
     * @param string $username
     * @param string $email
     * @param string $password
     * @param bool $isAdmin
     * @param string|null $expiryDate Optional expiry date string (YYYY-MM-DD HH:MM:SS) or null for permanent.
     * @return int|false The ID of the newly created user or false on failure.
     */
    public function create(string $username, string $email, string $password, bool $isAdmin = false, ?string $expiryDate = null) {
        // Check if username or email already exists
        if ($this->findByUsername($username) || $this->findByEmail($email)) {
            return false; // Or throw an exception
        }

        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        $status = $expiryDate ? (new DateTime($expiryDate) > new DateTime() ? 'active' : 'expired') : 'active'; // Initial status based on expiry
        if ($isAdmin) $status = 'active'; // Admins are always active initially

        $sql = "INSERT INTO users (username, email, password_hash, status, expiry_date, is_admin) 
                VALUES (:username, :email, :password_hash, :status, :expiry_date, :is_admin)";
        $stmt = $this->db->prepare($sql);

        $stmt->bindParam(":username", $username);
        $stmt->bindParam(":email", $email);
        $stmt->bindParam(":password_hash", $passwordHash);
        $stmt->bindParam(":status", $status);
        $stmt->bindParam(":expiry_date", $expiryDate); // Binds NULL if $expiryDate is null
        $stmt->bindValue(":is_admin", $isAdmin ? 1 : 0, PDO::PARAM_INT);

        if ($stmt->execute()) {
            return (int)$this->db->lastInsertId();
        }
        return false;
    }

    /**
     * Update user details (excluding password).
     *
     * @param int $id
     * @param string $username
     * @param string $email
     * @param string $status
     * @param string|null $expiryDate
     * @param bool $isAdmin
     * @return bool True on success, false on failure.
     */
    public function update(int $id, string $username, string $email, string $status, ?string $expiryDate, bool $isAdmin): bool {
        // Optional: Add checks to ensure new username/email don't conflict with others
        $sql = "UPDATE users SET 
                    username = :username, 
                    email = :email, 
                    status = :status, 
                    expiry_date = :expiry_date, 
                    is_admin = :is_admin 
                WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->bindParam(":username", $username);
        $stmt->bindParam(":email", $email);
        $stmt->bindParam(":status", $status);
        $stmt->bindParam(":expiry_date", $expiryDate);
        $stmt->bindValue(":is_admin", $isAdmin ? 1 : 0, PDO::PARAM_INT);

        return $stmt->execute();
    }

    /**
     * Update user password.
     *
     * @param int $id
     * @param string $newPassword
     * @return bool True on success, false on failure.
     */
    public function updatePassword(int $id, string $newPassword): bool {
        $passwordHash = password_hash($newPassword, PASSWORD_DEFAULT);
        $sql = "UPDATE users SET password_hash = :password_hash WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->bindParam(":password_hash", $passwordHash);
        return $stmt->execute();
    }

    /**
     * Delete a user.
     *
     * @param int $id
     * @return bool True on success, false on failure.
     */
    public function delete(int $id): bool {
        // Note: Foreign key constraints should handle related data (videos, etc.) if set to CASCADE
        $sql = "DELETE FROM users WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        return $stmt->execute();
    }

    /**
     * Get all users (primarily for admin panel).
     *
     * @return array List of all users.
     */
    public function getAll(): array {
        $stmt = $this->db->query("SELECT id, username, email, status, expiry_date, is_admin, created_at, updated_at FROM users ORDER BY created_at DESC");
        return $stmt->fetchAll();
    }

    /**
     * Check and update the status of expired users.
     * This could be run periodically (e.g., via cron or on admin login).
     *
     * @return int Number of users updated to 'expired'.
     */
    public function updateExpiredUsers(): int {
        $now = (new DateTime())->format('Y-m-d H:i:s');
        $sql = "UPDATE users SET status = 'expired' 
                WHERE status = 'active' 
                AND is_admin = 0 
                AND expiry_date IS NOT NULL 
                AND expiry_date < :now";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':now', $now);
        $stmt->execute();
        return $stmt->rowCount();
    }
}

